function majorityvoter()
% MAJORITYVOTER — 3D plot of the cross structure with N=4 STANDARD cells per branch,
% a central STANDARD SQUARE cell (L x L) and four layered corner fillers.
% Geometry matches stdcell_mv_gui when N=4

%% ==== Default parameters (nm) ====
P = struct( ...
    'L',24, 'W',10, 'H',2, ...                    % SiO2 base: X-length, Y-length, Z-thickness
    'origin',[0 0 0], ...                         % [x0 y0 z0]
    'H_Ti_bottom',1, ...                          % bottom Ti (Z)
    'H_gold_bottom',2, ...                        % bottom Au (Z)
    'L_vac',4, 'H_vac',3, ...                     % middle plane: vacuum width along X, Z-height
    'L_Au_top',16, 'W_Au_top',8, 'H_Au_top',4, ...% top cap size (X,Y,Z)
    'T_film',0.5, ...                             % Ti rim thickness
    'CapSide_center',22 ...                       % top square cap for the center block
);
N = 4;  % cells per branch (STANDARD)

%% ==== Colors / style ====
c_base   = [0.55 0.35 0.20];   % SiO2
c_ti     = [0.20 0.20 0.20];   % Ti
c_au     = [1.00 0.84 0.00];   % Au
c_vac    = [0.00 0.60 0.00];   % Vacuum
a_vac    = 0.35;               % alpha for Vacuum
c_hfo2   = [0.80 0.65 0.45];   % HfO2
edgeCol  = 'k'; 
lw       = 1.0;
F = faces6();

%% ==== Derived heights ====
H_tot  = P.H + P.H_Ti_bottom + P.H_gold_bottom + P.H_vac + P.H_Au_top;

%% ==== Build geometry ====
x0 = P.origin(1); 
y0 = P.origin(2); 
z0 = P.origin(3);

% Branch pitch (STANDARD cells): step in Y for Y-branches; in X for X-branches
pitch = P.W;

% SOUTH branch (before block): N STANDARD cells starting at y0
y_cursor = y0;
G_south = cell(1,N);
for i = 1:N
    G_south{i} = buildCellVerts(P, [x0, y_cursor, z0]);  % STANDARD
    y_cursor   = y_cursor + pitch;
end

% Center block: STANDARD SQUARE (L x L), trench full side, square top cap
oB  = [x0, y_cursor, z0];
Gc  = buildCentralSquareCellVerts(P, oB);
yB2 = oB(2) + P.L;   % end of central block along +Y

% NORTH branch (after block): N STANDARD cells
G_north = cell(1,N);
y_cursor = yB2;
for i = 1:N
    G_north{i} = buildCellVerts(P, [x0, y_cursor, z0]);  % STANDARD
    y_cursor   = y_cursor + pitch;
end

% Rotation center (for X-branches)
Crot = [oB(1) + P.L/2, oB(2) + P.L/2];

% Build rotated copies (+90° around Z): WEST (negative X) and EAST (positive X)
Rz = [0 -1 0; 1 0 0; 0 0 1];
G_south_r = cellfun(@(G) rotZ_around(G,Rz,Crot), G_south, 'UniformOutput', false);
G_north_r = cellfun(@(G) rotZ_around(G,Rz,Crot), G_north, 'UniformOutput', false);

% Corner fillers (4 layers, opaque), span = N*pitch
arm_span = N * pitch;
xB  = oB(1);        yB  = oB(2);
xB2 = xB + P.L;     yB2 = yB + P.L;

h1 = P.H + P.H_Ti_bottom;
h2 = P.H_gold_bottom;
h3 = P.H_vac;
h4 = P.H_Au_top;

o_NE = [xB2,            yB2,            z0];
o_NW = [xB - arm_span,  yB2,            z0];
o_SW = [xB - arm_span,  yB - arm_span,  z0];
o_SE = [xB2,            yB - arm_span,  z0];

FNE = buildFillerLayers(o_NE, arm_span, arm_span, [h1 h2 h3 h4]);
FNW = buildFillerLayers(o_NW, arm_span, arm_span, [h1 h2 h3 h4]);
FSW = buildFillerLayers(o_SW, arm_span, arm_span, [h1 h2 h3 h4]);
FSE = buildFillerLayers(o_SE, arm_span, arm_span, [h1 h2 h3 h4]);

%% ==== PLOT ====
fig = figure('Color','w','Name','Majority-Voter Cross (N=4, STANDARD)');
ax  = axes(fig); cla(ax,'reset'); hold(ax,'on');

% Draw SOUTH branch (legend shown once)
drawGeom(ax, G_south{1}, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, true);
for i=2:N, drawGeom(ax, G_south{i}, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false); end

% Draw CENTER block
drawGeom(ax, Gc, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false);

% Draw NORTH branch
for i=1:N, drawGeom(ax, G_north{i}, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false); end

% Draw rotated branches (WEST/EAST)
for i=1:N, drawGeom(ax, G_south_r{i}, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false); end
for i=1:N, drawGeom(ax, G_north_r{i}, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, false); end

% Draw corner fillers
drawFillerLayers(ax, FNE, c_base, c_au, c_hfo2, c_base, edgeCol, lw, F, 'FILLER (4 layers)');
drawFillerLayers(ax, FNW, c_base, c_au, c_hfo2, c_base, edgeCol, lw, F, '', true);
drawFillerLayers(ax, FSW, c_base, c_au, c_hfo2, c_base, edgeCol, lw, F, '', true);
drawFillerLayers(ax, FSE, c_base, c_au, c_hfo2, c_base, edgeCol, lw, F, '', true);

axis(ax,'equal'); grid(ax,'on'); view(ax,35,25);
xlabel(ax,'X [nm]'); ylabel(ax,'Y [nm]'); zlabel(ax,'Z [nm]');
title(ax, sprintf('Majority-Voter Cross — N=%d (ALL STANDARD) — H_{tot}=%.3g nm — Cap_{center}=%.3g nm', ...
                  N, H_tot, P.CapSide_center));

% Nice limits
branch_span = N * pitch;
total_span  = 2*branch_span + P.L;
marg   = 0.12 * [total_span, total_span, H_tot];
xlim(ax,[P.origin(1) - total_span/2 - marg(1), P.origin(1) + total_span/2 + marg(1)]);
ylim(ax,[P.origin(2) - marg(2),                P.origin(2) + total_span + marg(2)]);
zlim(ax,[P.origin(3) - marg(3),                P.origin(3) + H_tot + marg(3)]);

plot3(ax,P.origin(1),P.origin(2),P.origin(3),'ko','MarkerFaceColor','k','HandleVisibility','off');
legend(ax,'Location','bestoutside');
hold(ax,'off');

end % ==== END MAIN FUNCTION ====


%% ==================== GEOMETRY HELPERS ====================

function G = buildCellVerts(P, o)
% Standard rectangular cell (no CUT); returns a struct of vertex lists
V_base = vbox(o, P.L, P.W, P.H);
o_Ti   = o + [0 0 P.H];
V_Ti   = vbox(o_Ti, P.L, P.W, P.H_Ti_bottom);
o_Au   = o + [0 0 (P.H + P.H_Ti_bottom)];
V_Au   = vbox(o_Au, P.L, P.W, P.H_gold_bottom);

z0_top = o(3) + P.H + P.H_Ti_bottom + P.H_gold_bottom;
L_vac_eff = min(max(P.L_vac,0), P.L);
L_left  = max(0, (P.L - L_vac_eff)/2);
L_right = max(0, P.L - (L_left + L_vac_eff));

o_left  = [o(1),                 o(2), z0_top];
o_vac   = [o(1) + L_left,        o(2), z0_top];
o_right = [o(1) + L_left + L_vac_eff, o(2), z0_top];

V_left  = vbox(o_left,  L_left,    P.W, P.H_vac);
V_vac   = vbox(o_vac,   L_vac_eff, P.W, P.H_vac);
V_right = vbox(o_right, L_right,   P.W, P.H_vac);

x0_cap = o(1) + (P.L - P.L_Au_top)/2;
y0_cap = o(2) + (P.W - P.W_Au_top)/2;
z0_cap = z0_top + P.H_vac;
o_cap  = [x0_cap, y0_cap, z0_cap];
V_cap  = vbox(o_cap, P.L_Au_top, P.W_Au_top, P.H_Au_top);

o_film_xL = [x0_cap - P.T_film,      y0_cap,              z0_cap];
o_film_xR = [x0_cap + P.L_Au_top,    y0_cap,              z0_cap];
V_film_xL = vbox(o_film_xL, P.T_film,             P.W_Au_top, P.H_Au_top);
V_film_xR = vbox(o_film_xR, P.T_film,             P.W_Au_top, P.H_Au_top);
o_film_yF = [x0_cap - P.T_film,      y0_cap - P.T_film,   z0_cap];
o_film_yB = [x0_cap - P.T_film,      y0_cap + P.W_Au_top, z0_cap];
V_film_yF = vbox(o_film_yF, P.L_Au_top + 2*P.T_film, P.T_film,   P.H_Au_top);
V_film_yB = vbox(o_film_yB, P.L_Au_top + 2*P.T_film, P.T_film,   P.H_Au_top);

x_in_min = x0_cap - P.T_film; x_in_max = x0_cap + P.L_Au_top + P.T_film;
y_in_min = y0_cap - P.T_film; y_in_max = y0_cap + P.W_Au_top + P.T_film;

x_base_min = o(1); x_base_max = o(1) + P.L;
y_base_min = o(2); y_base_max = o(2) + P.W;

T_left_frame  = max(0, x_in_min - x_base_min);
T_right_frame = max(0, x_base_max - x_in_max);
T_front_frame = max(0, y_in_min - y_base_min);
T_back_frame  = max(0,  y_base_max - y_in_max);

z0_frame = z0_cap; H_frame = P.H_Au_top;

V_frame_xL = []; V_frame_xR = []; V_frame_yF = []; V_frame_yB = [];
if T_left_frame  > 0, V_frame_xL = vbox([x_base_min, y_base_min, z0_frame], T_left_frame,  P.W, H_frame); end
if T_right_frame > 0, V_frame_xR = vbox([x_in_max,   y_base_min, z0_frame], T_right_frame, P.W, H_frame); end
if T_front_frame > 0, V_frame_yF = vbox([x_in_min,   y_base_min, z0_frame], max(0,x_in_max-x_in_min), T_front_frame, H_frame); end
if T_back_frame  > 0, V_frame_yB = vbox([x_in_min,   y_in_max,   z0_frame], max(0,x_in_max-x_in_min), T_back_frame,  H_frame); end

G.base  = V_base;  G.ti   = V_Ti;    G.au   = V_Au;
G.left  = V_left;  G.vac  = V_vac;   G.right= V_right;
G.cap   = V_cap;   G.fxL  = V_film_xL; G.fxR = V_film_xR;
G.fyF   = V_film_yF; G.fyB = V_film_yB;
G.frmL  = V_frame_xL; G.frmR = V_frame_xR;
G.frmF  = V_frame_yF; G.frmB = V_frame_yB;
end

function G = buildCentralSquareCellVerts(P, o)
% Standard SQUARE center cell: side = P.L, trench = full side, top cap side = CapSide_center
Lside = P.L; Wside = P.L;

V_base = vbox(o, Lside, Wside, P.H);
o_Ti   = o + [0 0 P.H];
V_Ti   = vbox(o_Ti, Lside, Wside, P.H_Ti_bottom);
o_Au   = o + [0 0 (P.H + P.H_Ti_bottom)];
V_Au   = vbox(o_Au, Lside, Wside, P.H_gold_bottom);

z0_top = o(3) + P.H + P.H_Ti_bottom + P.H_gold_bottom;
V_left = []; V_right = [];
V_vac  = vbox([o(1) o(2) z0_top], Lside, Wside, P.H_vac);

capSide = min(P.CapSide_center, Lside);
x0_cap  = o(1) + (Lside - capSide)/2;
y0_cap  = o(2) + (Wside - capSide)/2;
z0_cap  = z0_top + P.H_vac;
V_cap   = vbox([x0_cap y0_cap z0_cap], capSide, capSide, P.H_Au_top);

V_film_xL = vbox([x0_cap - P.T_film, y0_cap,               z0_cap], P.T_film,             capSide, P.H_Au_top);
V_film_xR = vbox([x0_cap + capSide,  y0_cap,               z0_cap], P.T_film,             capSide, P.H_Au_top);
V_film_yF = vbox([x0_cap - P.T_film, y0_cap - P.T_film,    z0_cap], capSide + 2*P.T_film, P.T_film, P.H_Au_top);
V_film_yB = vbox([x0_cap - P.T_film, y0_cap + capSide,     z0_cap], capSide + 2*P.T_film, P.T_film, P.H_Au_top);

x_in_min = x0_cap - P.T_film; x_in_max = x0_cap + capSide + P.T_film;
y_in_min = y0_cap - P.T_film; y_in_max = y0_cap + capSide + P.T_film;
x_base_min = o(1); x_base_max = o(1) + Lside;
y_base_min = o(2); y_base_max = o(2) + Wside;

T_left_frame  = max(0, x_in_min - x_base_min);
T_right_frame = max(0, x_base_max - x_in_max);
T_front_frame = max(0, y_in_min - y_base_min);
T_back_frame  = max(0,  y_base_max - y_in_max);

z0_frame = z0_cap; H_frame = P.H_Au_top;

V_frame_xL = []; V_frame_xR = []; V_frame_yF = []; V_frame_yB = [];
if T_left_frame  > 0, V_frame_xL = vbox([x_base_min, y_base_min, z0_frame], T_left_frame,  Wside, H_frame); end
if T_right_frame > 0, V_frame_xR = vbox([x_in_max,   y_base_min, z0_frame], T_right_frame, Wside, H_frame); end
if T_front_frame > 0, V_frame_yF = vbox([x_in_min,   y_base_min, z0_frame], max(0,x_in_max-x_in_min), T_front_frame, H_frame); end
if T_back_frame  > 0, V_frame_yB = vbox([x_in_min,   y_in_max,   z0_frame], max(0,x_in_max-x_in_min), T_back_frame,  H_frame); end

G.base  = V_base;  G.ti   = V_Ti;    G.au   = V_Au;
G.left  = V_left;  G.vac  = V_vac;   G.right= V_right;
G.cap   = V_cap;   G.fxL  = V_film_xL; G.fxR = V_film_xR;
G.fyF   = V_film_yF; G.fyB = V_film_yB;
G.frmL  = V_frame_xL; G.frmR = V_frame_xR;
G.frmF  = V_frame_yF; G.frmB = V_frame_yB;
end

function Gt = rotZ_around(G, Rz, Cxy)
% Rotate all vertex groups around Z by matrix Rz with respect to center Cxy (x,y)
fn = fieldnames(G);
for i=1:numel(fn)
    V = G.(fn{i});
    if isempty(V), Gt.(fn{i}) = []; continue; end
    Vc = V; Vc(:,1)=V(:,1)-Cxy(1); Vc(:,2)=V(:,2)-Cxy(2);
    Vrot = (Rz * Vc.').'; Vrot(:,1)=Vrot(:,1)+Cxy(1); Vrot(:,2)=Vrot(:,2)+Cxy(2);
    Gt.(fn{i}) = Vrot;
end
end

function drawGeom(ax,G, c_base,c_ti,c_au,c_vac,a_vac,c_hfo2,edgeCol,lw, F, addLegend)
% Draw all parts of one cell geometry
drawPatch(ax,G.base,  c_base, edgeCol,lw, addLegend, 'SiO2',  1,      F);
drawPatch(ax,G.ti,    c_ti,   edgeCol,lw, addLegend, 'Ti',    1,      F);
drawPatch(ax,G.au,    c_au,   edgeCol,lw, addLegend, 'Au',    1,      F);
drawPatch(ax,G.left,  c_hfo2, edgeCol,lw, addLegend, 'HfO2',  1,      F);
drawPatch(ax,G.vac,   c_vac,  edgeCol,lw, addLegend, 'vacuum',a_vac,  F);
drawPatch(ax,G.right, c_hfo2, edgeCol,lw, false,     '',      1,      F, true);
drawPatch(ax,G.cap,   c_au,   edgeCol,lw, false,     '',      1,      F, true);
drawPatch(ax,G.fxL,   c_ti,   edgeCol,lw, false,     '',      1,      F, true);
drawPatch(ax,G.fxR,   c_ti,   edgeCol,lw, false,     '',      1,      F, true);
drawPatch(ax,G.fyF,   c_ti,   edgeCol,lw, false,     '',      1,      F, true);
drawPatch(ax,G.fyB,   c_ti,   edgeCol,lw, false,     '',      1,      F, true);
drawPatch(ax,G.frmL,  c_base, edgeCol,lw, false,     '',      1,      F, true);
drawPatch(ax,G.frmR,  c_base, edgeCol,lw, false,     '',      1,      F, true);
drawPatch(ax,G.frmF,  c_base, edgeCol,lw, false,     '',      1,      F, true);
drawPatch(ax,G.frmB,  c_base, edgeCol,lw, false,     '',      1,      F, true);
end

function FL = buildFillerLayers(o, Lx, Wy, hvec)
% Build vertices for a 4-layer square filler of side 'span' starting at origin 'o'
z = o(3);
FL.V1 = vbox([o(1) o(2) z], Lx, Wy, hvec(1)); z = z + hvec(1);
FL.V2 = vbox([o(1) o(2) z], Lx, Wy, hvec(2)); z = z + hvec(2);
FL.V3 = vbox([o(1) o(2) z], Lx, Wy, hvec(3)); z = z + hvec(3);
FL.V4 = vbox([o(1) o(2) z], Lx, Wy, hvec(4));
end

function V = vbox(o,Lx,Wy,Hz)
% Create vertex list (8 vertices) for an axis-aligned cuboid
V = [ o;
      o + [Lx 0 0];
      o + [0 Wy 0];
      o + [0 0 Hz];
      o + [Lx Wy 0];
      o + [Lx 0 Hz];
      o + [0 Wy Hz];
      o + [Lx Wy Hz] ];
end

function F = faces6()
% Quad faces for the 8-vertex cuboid
F = [1 2 5 3; 3 5 8 7; 1 3 7 4; 2 6 8 5; 1 4 6 2; 4 7 8 6];
end

function ph = drawPatch(axh,V,col,edgeCol,lw, addToLegend, name, faceAlpha, F, offLegend)
% Draw a patch, optionally adding it to legend
if nargin<10, offLegend=false; end
if isempty(V), ph = []; return; end
if addToLegend && ~offLegend && ~isempty(name)
    ph = patch(axh,'Vertices',V,'Faces',F,'FaceColor',col,'FaceAlpha',faceAlpha, ...
               'EdgeColor',edgeCol,'LineWidth',lw,'DisplayName',name);
else
    ph = patch(axh,'Vertices',V,'Faces',F,'FaceColor',col,'FaceAlpha',faceAlpha, ...
               'EdgeColor',edgeCol,'LineWidth',lw,'HandleVisibility','off');
end
end

function drawFillerLayers(ax, FL, c1, c2, c3, c4, edgeCol, lw, F, legendName, offLegend)
% Draw a 4-layer filler (all opaque). Only layer 1 can appear in the legend.
if nargin < 10, legendName = ''; end
if nargin < 11, offLegend = false; end
% Layer 1 (legend if requested)
if ~isempty(legendName) && ~offLegend
    patch(ax,'Vertices',FL.V1,'Faces',F,'FaceColor',c1, ...
        'EdgeColor',edgeCol,'LineWidth',lw,'DisplayName',legendName);
else
    patch(ax,'Vertices',FL.V1,'Faces',F,'FaceColor',c1, ...
        'EdgeColor',edgeCol,'LineWidth',lw,'HandleVisibility','off');
end
% Layers 2–4 (no legend)
patch(ax,'Vertices',FL.V2,'Faces',F,'FaceColor',c2, ...
    'EdgeColor',edgeCol,'LineWidth',lw,'HandleVisibility','off');
patch(ax,'Vertices',FL.V3,'Faces',F,'FaceColor',c3, ...
    'EdgeColor',edgeCol,'LineWidth',lw,'HandleVisibility','off');
patch(ax,'Vertices',FL.V4,'Faces',F,'FaceColor',c4, ...
    'EdgeColor',edgeCol,'LineWidth',lw,'HandleVisibility','off');
end
